const { DB } = require("../../db");
const Router = require("express").Router();
const { verifySession } = require("../../middleware/verifySession");
const { ACCEPTED_FUND_CARD_SCHEMA } = require("../../validators/card");

// user request card
Router.post("/request/card", verifySession, async (req, res) => {
  try {
    const findCard = await DB.CARD.findOne({ uid: req.user._id }); //update here by aura

    if (findCard) {
      if (!findCard.cvv)
        throw new Error("You already have a pending card awaiting approval");

      res.status(500).json({
        msg: "You can't request another card; you already have one",
      });
    }

    // create the card now
    await DB.CARD.create({
      uid: req.user._id,
      status: "pending",
      history: [],
    });

    await DB.NOTIFICATION.create({
      uid: req.user._id,
      type: "alert",
      body: {
        title: "Pending Approval",
        // READING AND REPLACING WHERE {{NAME FOR THE REAL NAME}} ALL THE FILE SYSTEM
        text: "Your card is pending approval. We’ll notify you once it’s reviewed.",
      },
    });

    res.status(200).json({
      data: "Successful",
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request.",
    });
  }
});

// deleting card of any user details by user
Router.delete("/delete/card/:cardID", verifySession, async (req, res) => {
  try {
    const findCard = await DB.CARD.findById(req.params.cardID);
    if (!findCard) throw new Error("Can't find user with the required ID");

    // check if the card has amount and use the amount and add to the user usd balance
    if (findCard.amount > 0) {
      await DB.USER.findByIdAndUpdate(
        findCard.uid,
        {
          $inc: {
            "account.balance.usd": findCard.amount,
          },
        },
        { new: true }
      );
    }

    await DB.CARD.findByIdAndDelete(findCard._id);

    res.status(201).json({
      data: "Card deleted successfully",
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// user adding funds to the card from there usd balance
Router.put("/fund/card", verifySession, async (req, res) => {
  try {
    let { error, value } = ACCEPTED_FUND_CARD_SCHEMA.validate(req.body);
    if (error) throw new Error(error.details[0].message);

    let { amount } = value;

    // find the card
    const findCard = await DB.CARD.findOne({ uid: req.user._id });

    if (!findCard) throw new Error("Can't fund card, you do not have any");

    if (amount > req.user.account.balance.usd)
      throw new Error("Insufficient balance");

    // now minus the amount from the user usd balance
    await DB.USER.findByIdAndUpdate(
      req.user._id,
      {
        $inc: {
          "account.balance.usd": -amount,
        },
      },
      { new: true }
    );

    // now fund the card
    await DB.CARD.findByIdAndUpdate(
      findCard._id,
      {
        amount,
      },
      { new: true }
    );

    await DB.NOTIFICATION.create({
      uid: req.user._id,
      type: "info",
      body: {
        title: "Card Funded Successfully!",
        // READING AND REPLACING WHERE {{NAME FOR THE REAL NAME}} ALL THE FILE SYSTEM
        text: `Your card has been successfully funded. You can now start using it for your transactions. Thank you for choosing us!`,
      },
    });

    res.status(201).json({
      data: `Card funded ${amount} USD successfully`,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

module.exports = Router;
