const Router = require("express").Router();
const { DB } = require("../../db");
const { verifySession } = require("../../middleware/verifySession");
const { verifyAdmin } = require("../../middleware/verifyAdmin");
const {
  EDIT_ACCOUNT_SCHEMA,
  CHANGE_SUPPORT,
} = require("../../validators/account");
const { BCRYPT } = require("../../helper/bcrypt");
const { SUSPENDED_SCHEMA } = require("../../validators/suspend");
const fs = require("fs");
const path = require("path");

// admin change tawkto route
Router.post("/change/tawkto", verifySession, verifyAdmin, async (req, res) => {
  try {
    let { error, value } = CHANGE_SUPPORT.validate(req.body);
    if (error) throw new Error(error.details[0].message);

    const { support } = value;

    fs.writeFileSync(
      path.join(process.cwd() + "/messages/tawkto.txt"),
      support
    );

    res.status(201).json({
      data: "successful",
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request.",
    });
  }
});

// su get all users
Router.get("/list/users", verifySession, verifyAdmin, async (req, res) => {
  try {
    const allUsers = await DB.USER.find();

    res.status(201).json({
      data: allUsers,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request.",
    });
  }
});

// su modify user details
Router.put(
  "/modify/user/details/:id",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      let { error, value } = EDIT_ACCOUNT_SCHEMA.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      const findUser = await DB.USER.findById(req.params.id);
      if (!findUser) throw new Error("invalid user provided");

      if (value.password) {
        const hashed = await BCRYPT.hashPassword(value.password);
        findUser.password = hashed;
      }

      findUser.email = value.email;
      // findUser.phone = value.phone
      findUser.type = value.type;
      findUser.name = value.name;
      findUser.account = { ...findUser.account, balance: {
        usd: parseFloat(value.balance.usd),
        gbp: parseFloat(value.balance.gbp),
        euro: parseFloat(value.balance.euro),
      } };
      findUser.transaction_status = value.transaction_status
      findUser.failed_transaction_msg = value.failed_transaction_msg
      findUser.vat.code = value.vat_code || findUser.vat.code,
      findUser.vat.status = value.vat_status || findUser.vat.status

      await findUser.save();

      if(value.vat_status || findUser.vat.status){
        await DB.USER.updateOne({_id:req.params.id}, {
          'vat.code': value.vat_code || findUser.vat.code,
          'vat.status': value.vat_status || findUser.vat.status,
        })
      }

      res.status(201).json({
        data: "data updated successfully",
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request.",
      });
    }
  }
);

// su modify user password
Router.put(
  "/change/user/password/:id",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      let { error, value } = EDIT_ACCOUNT_SCHEMA.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      const abcd = await DB.USER.findByIdAndUpdate(
        req.params.id,
        { password: await BCRYPT.hashPassword(value.password) },
        { new: true }
      );

      res.status(201).json({
        data: abcd,
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request.",
      });
    }
  }
);

// su add to suspend or remove
Router.put(
  "/suspend/manage/:id",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findUser = await DB.USER.findById(req.params.id);
      if (!findUser) throw new Error("invalid user provided");

      let { error, value } = SUSPENDED_SCHEMA.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      // Check if the function is already in the suspended array
      const index = findUser.suspended.indexOf(value.suspended);

      if (index === -1) {
        // If not found, add the function to the suspended array
        findUser.suspended.push(value.suspended);
      } else {
        // If found, remove the function from the suspended array
        findUser.suspended.splice(index, 1);
      }

      // Save the updated user document
      await findUser.save();

      return res.status(201).json({
        message: `${value.suspended} has been updated`,
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request.",
      });
    }
  }
);

// su deleting user
Router.delete(
  "/delete/user/:id",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findUser = await DB.USER.findById(req.params.id);
      if (!findUser) throw new Error("invalid user provided");

      await DB.USER.findByIdAndDelete(findUser._id);

      res.status(201).json({
        message: `${findUser.name.first} has been deleted`,
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request.",
      });
    }
  }
);

// su getting all suspended user
Router.get(
  "/suspend/accounts",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const foundUser = await DB.USER.find({ suspended: { $ne: [] } })
        .select("email name suspended _id")
        .sort({ createdAt: -1 });

      res.status(201).json({
        data: foundUser,
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request.",
      });
    }
  }
);

// su lifting all user suspension
Router.get(
  "/lift/suspension/:id",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findUser = await DB.USER.findById(req.params.id);
      if (!findUser) throw new Error("invalid user provided");

      await DB.USER.findByIdAndUpdate(findUser, {
        suspended: [],
      });

      const foundUser = await DB.USER.find({ suspended: { $ne: [] } })
        .select("email name suspended _id")
        .sort({ createdAt: -1 });

      res.status(201).json({
        data: foundUser,
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request.",
      });
    }
  }
);

module.exports = Router;
