const { Schema, model, mongoose } = require("mongoose");

const beneficiarySchema = new Schema(
  {
    // Reference to the user who owns this beneficiary
    uid: { 
      type: mongoose.Schema.Types.ObjectId, 
      ref: "User", 
      required: true 
    },
    
    // Basic beneficiary info
    name: { 
      type: String, 
      required: true,
      trim: true
    },
    
    // Transfer type (local or wire)
    type: { 
      type: String, 
      enum: ["local", "wire"], 
      required: true 
    },
    
    // Common fields for both types
    accountNumber: { 
      type: String, 
      required: true,
      trim: true
    },
    bankName: { 
      type: String, 
      required: true,
      trim: true
    },
    
    // Fields specific to local (SEPA) transfers
    localDetails: {
      routingNumber: String,  // For US/Canada
      sortCode: String,      // For UK
      iban: String,         // For EU
      bic: String           // Bank Identifier Code
    },
    
    // Fields specific to international (Wire) transfers
    wireDetails: {
      swiftCode: String,     // SWIFT/BIC code
      bankAddress: String,
      beneficiaryAddress: String,
      country: {
        type: String,
        enum: [
  'AF', 'AX', 'AL', 'DZ', 'AS', 'AD', 'AO', 'AI', 'AQ', 'AG', 'AR', 'AM', 'AW',
  'AU', 'AT', 'AZ', 'BS', 'BH', 'BD', 'BB', 'BY', 'BE', 'BZ', 'BJ', 'BM', 'BT',
  'BO', 'BQ', 'BA', 'BW', 'BV', 'BR', 'IO', 'BN', 'BG', 'BF', 'BI', 'CV', 'KH',
  'CM', 'CA', 'KY', 'CF', 'TD', 'CL', 'CN', 'CX', 'CC', 'CO', 'KM', 'CG', 'CD',
  'CK', 'CR', 'CI', 'HR', 'CU', 'CW', 'CY', 'CZ', 'DK', 'DJ', 'DM', 'DO', 'EC',
  'EG', 'SV', 'GQ', 'ER', 'EE', 'SZ', 'ET', 'FK', 'FO', 'FJ', 'FI', 'FR', 'GF',
  'PF', 'TF', 'GA', 'GM', 'GE', 'DE', 'GH', 'GI', 'GR', 'GL', 'GD', 'GP', 'GU',
  'GT', 'GG', 'GN', 'GW', 'GY', 'HT', 'HM', 'VA', 'HN', 'HK', 'HU', 'IS', 'IN',
  'ID', 'IR', 'IQ', 'IE', 'IM', 'IL', 'IT', 'JM', 'JP', 'JE', 'JO', 'KZ', 'KE',
  'KI', 'KP', 'KR', 'KW', 'KG', 'LA', 'LV', 'LB', 'LS', 'LR', 'LY', 'LI', 'LT',
  'LU', 'MO', 'MK', 'MG', 'MW', 'MY', 'MV', 'ML', 'MT', 'MH', 'MQ', 'MR', 'MU',
  'YT', 'MX', 'FM', 'MD', 'MC', 'MN', 'ME', 'MS', 'MA', 'MZ', 'MM', 'NA', 'NR',
  'NP', 'NL', 'NC', 'NZ', 'NI', 'NE', 'NG', 'NU', 'NF', 'MP', 'NO', 'OM', 'PK',
  'PW', 'PS', 'PA', 'PG', 'PY', 'PE', 'PH', 'PN', 'PL', 'PT', 'PR', 'QA', 'RE',
  'RO', 'RU', 'RW', 'BL', 'SH', 'KN', 'LC', 'MF', 'PM', 'VC', 'WS', 'SM', 'ST',
  'SA', 'SN', 'RS', 'SC', 'SL', 'SG', 'SX', 'SK', 'SI', 'SB', 'SO', 'ZA', 'GS',
  'SS', 'ES', 'LK', 'SD', 'SR', 'SJ', 'SE', 'CH', 'SY', 'TW', 'TJ', 'TZ', 'TH',
  'TL', 'TG', 'TK', 'TO', 'TT', 'TN', 'TR', 'TM', 'TC', 'TV', 'UG', 'UA', 'AE',
  'GB', 'US', 'UM', 'UY', 'UZ', 'VU', 'VE', 'VN', 'VG', 'VI', 'WF', 'EH', 'YE',
  'ZM', 'ZW'
],
        uppercase: true
      },
      intermediaryBank: {
        name: String,
        account: String,
        swiftCode: String
      }
    },
    
    // Metadata
    isFavorite: { 
      type: Boolean, 
      default: false 
    },
    lastUsed: Date,
    
    // Additional info
    email: {
      type: String,
      trim: true,
      lowercase: true,
      validate: {
        validator: function(v) {
          return /^\w+([\.-]?\w+)*@\w+([\.-]?\w+)*(\.\w{2,3})+$/.test(v);
        },
        message: props => `${props.value} is not a valid email address!`
      }
    },
    phone: String,
    
    // For internal use
    notes: String,
    tags: [String]
  },
  {
    timestamps: true,
    toJSON: { virtuals: true },
    toObject: { virtuals: true }
  }
);

// Indexes for faster querying
beneficiarySchema.index({ user: 1, type: 1 });
beneficiarySchema.index({ user: 1, isFavorite: 1 });
beneficiarySchema.index({ user: 1, lastUsed: -1 });

// Virtual for displaying formatted account number
beneficiarySchema.virtual('displayAccount').get(function() {
  return `•••• ${this.accountNumber?.slice(-4)}`;
});

// Pre-save hook to normalize data
beneficiarySchema.pre('save', function(next) {
  // Remove spaces from account numbers
  if (this.accountNumber) {
    this.accountNumber = this.accountNumber.replace(/\s/g, '');
  }
  
  // For IBANs, ensure uppercase
  if (this.localDetails?.iban) {
    this.localDetails.iban = this.localDetails.iban.toUpperCase().replace(/\s/g, '');
  }
  
  // For SWIFT codes, ensure uppercase
  if (this.wireDetails?.swiftCode) {
    this.wireDetails.swiftCode = this.wireDetails.swiftCode.toUpperCase().replace(/\s/g, '');
  }
  
  next();
});

// Static method for finding by user
beneficiarySchema.statics.findByUser = function(userId, type) {
  return this.find({ user: userId, ...(type && { type }) })
             .sort({ isFavorite: -1, lastUsed: -1 });
};

module.exports = model("Beneficiary", beneficiarySchema);