const Router = require("express").Router();
const { DB } = require("../../db");
const { verifySession } = require("../../middleware/verifySession");
const { verifyAdmin } = require("../../middleware/verifyAdmin");
const { EDIT_TRANSACTION } = require("../../validators/suspend");
const { default: mongoose } = require("mongoose");
const { sendTransactionAlertEmail } = require("../../helper/sendEmail");

// SU getting all transaction list
Router.get(
  "/get/all/transaction",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const allTransactions = await DB.TRANSACTION.find().sort({createdAt: -1}).populate("uid", "image email name");

      res.status(201).json({ data: allTransactions });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// SU getting any all transaction for any particular user
Router.get(
  "/get/all/transaction/:userID",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const allTransactions = await DB.TRANSACTION.find({
        uid: req.params.userID,
      });

      res.status(201).json({ data: allTransactions });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);


Router.put(
  "/edit/transaction/:id",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      // Validate request body
      const { error, value } = EDIT_TRANSACTION.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      // Find transaction
      const transaction = await DB.TRANSACTION.findById(req.params.id);
      if (!transaction) throw new Error("Transaction not found");

      // Prevent modifying completed transactions
      if (transaction.status === 'successful') {
        throw new Error("Cannot modify already completed transaction");
      }

      // Update transaction
      const updatedTransaction = await DB.TRANSACTION.findByIdAndUpdate(
        req.params.id,
        { ...value },
        { new: true }
      );

      // Process balance updates if status changed to completed
      if (value.status === 'successful' && transaction.status !== 'successful') {
        const amount = transaction.amount;
        const currency = transaction.transfer_from.toLowerCase() == 'eur' ? 'euro' : transaction.transfer_from;
        const userId = transaction.uid;

        if (transaction.type === 'debit') {
          // Debit transaction - subtract from balance
          await DB.USER.updateOne(
            { _id: userId },
            { $inc: { [`account.balance.${currency}`]: -amount } }
          );
        } else if (transaction.type === 'credit') {
          // Credit transaction - add to balance
          await DB.USER.updateOne(
            { _id: userId },
            { $inc: { [`account.balance.${currency}`]: amount } }
          );
        }
      }

      res.status(200).json({
        success: true,
        data: updatedTransaction,
        msg: "Transaction updated successfully"
      });

    } catch (error) {
      console.error("Transaction update error:", error);
      res.status(500).json({
        success: false,
        msg: error.message || "Failed to update transaction"
      });
    }
  }
);

const updateUserBalance = async (userId, currency, amount, options = {}) => {
  const { type, checkBalance = false } = options;
  const currencyPath = `account.balance.${currency}`;
  console.log('====================================');
  console.log(currencyPath);
  console.log('====================================');
  const amountNum = Number(amount);

  if (isNaN(amountNum)) {
    throw { status: 400, message: 'Invalid amount', success: false };
  }

  const session = await mongoose.startSession();
  try {
    session.startTransaction();

    // 1. First ensure the field exists and is numeric (using regular number conversion)
    await DB.USER.updateOne(
      { _id: userId },
      [
        {
          $set: {
            [currencyPath]: {
              $let: {
                vars: {
                  currentValue: {
                    $ifNull: [
                      `$${currencyPath}`,
                      type === 'credit' ? amountNum : 0
                    ]
                  }
                },
                in: {
                  $cond: [
                    { $eq: [{ $type: "$$currentValue" }, "string"] },
                    { $toDouble: "$$currentValue" },
                    {
                      $cond: [
                        { $eq: [{ $type: "$$currentValue" }, "decimal"] },
                        { $toDouble: "$$currentValue" },
                        "$$currentValue"
                      ]
                    }
                  ]
                }
              }
            }
          }
        }
      ],
      { session }
    );

    // 2. Handle debit with balance check
    if (type === 'debit' && checkBalance) {
      const updateResult = await DB.USER.updateOne(
        { 
          _id: userId,
          [currencyPath]: { $gte: amountNum }
        },
        { $inc: { [currencyPath]: -amountNum } },
        { session }
      );

      if (updateResult.matchedCount === 0) {
        throw { status: 400, message: 'Insufficient balance', success: false };
      }
    } 
    // 3. Handle credit or debit without check
    else {
      await DB.USER.updateOne(
        { _id: userId },
        { $inc: { [currencyPath]: amountNum } },
        { session }
      );
    }

    const updatedUser = await DB.USER.findById(userId).session(session);
    await session.commitTransaction();
    return updatedUser;

  } catch (error) {
    await session.abortTransaction();
    console.error('Balance update failed:', {
      userId,
      currency,
      error: error.message,
      stack: error.stack
    });
    throw error.status ? error : { 
      status: 500, 
      message: 'Database operation failed',
      success: false,
      dbError: error.message
    };
  } finally {
    session.endSession();
  }
};


Router.post('/create/transaction', verifySession, verifyAdmin, async (req, res) => {
  try {
    const {
      user_id,
      amount,
      memo = '',
      transfer_type,
      transfer_from,
      type,
      status
    } = req.body;

    // Validate required fields
    if (!user_id || !amount || !transfer_type || !transfer_from || !type || !status) {
      return res.status(400).json({ 
        success: false,
        msg: 'Missing required fields' 
      });
    }

    // Validate amount is positive number
    const amountNum = Number(amount);
    if (isNaN(amountNum) || amountNum <= 0) {
      return res.status(400).json({ 
        success: false,
        msg: 'Amount must be a positive number' 
      });
    }

    // Validate transaction type
    const validTypes = ['credit', 'debit'];
    if (!validTypes.includes(type)) {
      return res.status(400).json({ 
        success: false,
        msg: 'Invalid transaction type' 
      });
    }

    // Validate transfer type
    const validTransferTypes = ['wire', 'local'];
    if (!validTransferTypes.includes(transfer_type)) {
      return res.status(400).json({ 
        success: false,
        msg: 'Invalid transfer type' 
      });
    }

    // Validate currency
    const validCurrencies = ['usd', 'euro', 'gbp'];
    if (!validCurrencies.includes(transfer_from)) {
      return res.status(400).json({ 
        success: false,
        msg: 'Invalid currency' 
      });
    }

    // Validate status
    const validStatuses = ["pending", "successful", "rejected", "reversed"];
    if (!validStatuses.includes(status)) {
      return res.status(400).json({ 
        success: false,
        msg: 'Invalid status' 
      });
    }

    // Check if user exists
    const userExists = await DB.USER.findOne({ _id: user_id });
    if (!userExists) {
      return res.status(404).json({ 
        success: false,
        msg: 'User not found' 
      });
    }

    // Create transaction reference
    const ref = `TRX-${Date.now().toString().slice(-6)}-${Math.floor(Math.random() * 9000) + 1000}`;

    // Create the transaction
    const newTransaction = new DB.TRANSACTION({
      uid: user_id,
      amount: amountNum,
      memo,
      transfer_type,
      transfer_from:transfer_from.replace('o',''),
      type,
      status,
      ref,
      receiverID: user_id,
      metadata: {
        createdBy: req.user.id,
        beneficiaryName: "Recipient Name" // You might want to fetch this from user data
      }
    });

    // Save transaction
    await newTransaction.save();

    // Update user balance if transaction is successful
if (status === 'successful') {
  try {
    // Use the improved updateUserBalance function
    await updateUserBalance(
      user_id,
      transfer_from,
      amountNum,
      {
        type: type,  // 'credit' or 'debit'
        checkBalance: type === 'debit',
        upsert: type === 'credit'
      }
    );

    // Send email notification
    await sendTransactionAlertEmail(
      userExists.name.first,
      userExists.email,
      type,
      amount,
      transfer_from.replace('o',''),
      '*****',
      new Date(),
      ref
    );

  } catch (error) {

    console.error(error);
    
    // Handle insufficient balance error
    if (error.status === 400) {
      return res.status(400).json({ 
        success: false,
        msg: error.message 
      });
    }
    // Handle other errors
    return res.status(500).json({
      success: false,
      msg: 'Failed to update balance'
    });
  }
}


    res.status(201).json({
      success: true,
      msg: 'Transaction created successfully',
      data: newTransaction
    });

  } catch (err) {
    console.error('Error creating transaction:', err);
    res.status(500).json({ 
      success: false,
      msg: err.message || 'Server error' 
    });
  }
});

// delete trx 
Router.delete('/delete/transaction/:trx_id', verifySession, verifyAdmin, async(req,res)=>{
    try {
      await DB.TRANSACTION.deleteOne({_id: req.params.trx_id})
      res.status(201).json({
        data: 'Transaction deleted'
      })
    } catch (error) {
      console.error(error);
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
})







module.exports = Router;
