
const { Schema, model, default: mongoose } = require("mongoose");
const moment = require("moment");

/**
 * Transaction Schema
 * Represents financial transactions between users
 */
const TransactionModel = new Schema(
  {
    uid: { 
      type: mongoose.Schema.Types.ObjectId, 
      ref: "users",
      required: true,
      index: true 
    },
    ref: {
      type: String,
      unique: true,
      default: () => `TRX-${moment().format('YYYYMMDD')}-${Math.floor(1000 + Math.random() * 9000)}`
    },
    receiverID: { 
      type: mongoose.Schema.Types.ObjectId, 
      ref: "beneficiaries",
      required: true
    },
    amount: {
      type: Number,
      required: true,
      min: 0.01,
      set: v => parseFloat(v.toFixed(2)) // Ensure 2 decimal places
    },
    memo: {
      type: String,
      maxlength: 200,
      trim: true
    },
    repeat: {
      type: {
        frequency: {
          type: String,
          enum: ["daily", "weekly", "monthly", "quarterly", "yearly"]
        },
        duration: {
          type: String,
          enum: ["1-month", "3-months", "6-months", "1-year", "indefinite"]
        },
        next_run: Date
      },
      required: false
    },
    transfer_type: { 
      type: String, 
      enum: ["local", "wire"], 
      default: "local",
      index: true
    },
    transfer_from: { 
      type: String, 
      enum: ["usd", "gbp", "eur"], 
      default: "usd",
    },
    charges: {
      type: Number,
      default: 0,
      min: 0
    },
    status: {
      type: String,
      enum: ["pending", "successful", "rejected", "reversed"],
      default: "pending"
    },
    type: {
      type: String,
      enum: ["credit", "debit"],
      required: true
    },
    metadata: Schema.Types.Mixed // For additional data
  },
  {
    timestamps: true,
    toJSON: { virtuals: true },
    toObject: { virtuals: true }
  }
);

// Indexes for faster queries
TransactionModel.index({ uid: 1, status: 1 });
TransactionModel.index({ receiverID: 1, createdAt: -1 });

// Virtual for formatted date
TransactionModel.virtual('date_formatted').get(function() {
  return moment(this.createdAt).format('MMM Do YYYY, h:mm:ss a');
});

// Pre-save hook to update status based on user's transaction_status
TransactionModel.pre('save', async function(next) {
  if (this.isNew) {
    const user = await mongoose.model('users').findById(this.uid);
    if (user) {
      if (user.transaction_status === 'pending') {
        this.status = 'pending';
      } else if (user.transaction_status === 'fail') {
        this.status = 'rejected';
      }
    }else{
      console.log('users else');
      
    }
  }
  next();
});


TransactionModel.statics = {
  async updateStatus(transactionId, newStatus, adminId, reason = '') {
    const validStatuses = ['pending', 'successful', 'rejected', 'reversed'];
    if (!validStatuses.includes(newStatus)) {
      throw new Error('Invalid status');
    }

    const transaction = await this.findById(transactionId);
    if (!transaction) throw new Error('Transaction not found');

    // Status transition validation
    const validTransitions = {
      pending: ['successful', 'rejected'],
      successful: ['reversed'],
      rejected: [],
      reversed: []
    };

    if (!validTransitions[transaction.status].includes(newStatus)) {
      throw new Error(`Cannot change status from ${transaction.status} to ${newStatus}`);
    }

    // Update transaction
    transaction.status = newStatus;
    transaction.metadata = {
      ...transaction.metadata,
      statusUpdatedBy: adminId,
      statusUpdateReason: reason,
      statusUpdatedAt: new Date()
    };

    // Handle reversals
    if (newStatus === 'reversed' && transaction.status === 'successful') {
      await Promise.all([
        User.findByIdAndUpdate(transaction.uid, { $inc: { balance: transaction.amount } }),
        User.findByIdAndUpdate(transaction.receiverID, { $inc: { balance: -transaction.amount } })
      ]);
    }

    return transaction.save();
  }
};

module.exports = model("Transactions", TransactionModel);








// const { Schema, model, default: mongoose } = require("mongoose");
// const moment = require("moment");

// const TransactionModel = new Schema(
//   {
//     uid: { type: mongoose.Schema.Types.ObjectId, ref: "users" },
//     // balance_after_debit: String,
//     ref: String,
//     bankID: { type: mongoose.Schema.Types.ObjectId, ref: "bank_accounts" },
//     receiverID: { type: mongoose.Schema.Types.ObjectId, ref: "users" },
//     amount: Number,
//     memo: String, //narration of transfer 
//     repeat: { type: Object, 
//       frequency: String,
//       duration: String,
//      },
//     transfer_type: { type: String, enum: ["local", "wire"], default: "local" },
//     transfer_from: { type: String, enum: ["usd", "gbp", "euro"], default: "usd" },
//     charges: Number,
//     date: { type: Date, default: Date.now },
//     status: {
//       type: String,
//       enum: ["pending", "successful", 'rejected', 'reversed'],
//       default: "successful",
//     },
//     type: {
//       type: String,
//       enum: ['credit','debit'],
//       default: "credit",
//     },
//   },
//   {
//     timestamps: true,
//   }
// );

// module.exports = model("Transactions", TransactionModel);
