const { default: mongoose } = require("mongoose");
const { DB } = require("../../db");
const {  sendOTPEmail, sendTransactionAlertEmail } = require("../../helper/sendEmail");
const { sendSMS } = require("../../helper/sendSMS");
const { verifySession } = require("../../middleware/verifySession");
const { transferSchema } = require("../../validators/transaction");
const {
  ACCEPTED_TRANSFER_SCHEMA,
  FIND_BANK_SCHEMA,
} = require("../../validators/transfer.validator");
const speakeasy = require("speakeasy");
const Router = require("express").Router();

const generateRef = (length) => {
  var a =
    "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890".split("");
  var b = [];
  for (var i = 0; i < length; i++) {
    var j = (Math.random() * (a.length - 1)).toFixed(0);
    b[i] = a[j];
  }
  return b.join("")?.toLowerCase();
};

// user send otp here! for sending transactions for only email and phone number
Router.post("/send/otp", verifySession, async (req, res) => {
  try {
    const findUser = await DB.USER.findById(req.user._id);
    if (!findUser) throw new Error("Can't find user with the required ID");

    if (findUser.tfa.status == "inactive")
      throw new Error("Enable two factor to continue");

    const code = Math.random().toString(20).split("").splice(2, 6).join("");

    if (findUser.otp.direction == "email") {
      findUser.otp = { code, direction: "email" };

      await findUser.save();
      sendOTPEmail( findUser.name.first, findUser.email, code);

      res.json({
        data: "successful",
      });
      return;
    }

    if (findUser.otp.direction == "phone") {
      findUser.otp = { code, direction: "phone" };

      await findUser.save();
      const text = `Your OTP for verification is ${code}. Use this code to authorize the transaction on your account.`;
      await sendSMS(findUser.phone, text);

      res.json({
        data: "successful",
      });
      return;
    }

    res.json({
      data: "success",
    });

  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

// GET ACCOUNT DETAILS
Router.post("/find/account", verifySession, async (req, res) => {
  try {
    let { error, value } = FIND_BANK_SCHEMA.validate(req.body);
    if (error) throw new Error(error.details[0].message);

    let { account_number, routing_number, type } = value;

    if (type == "wire") {
      const account = await DB.BANKS.findOne({
        account_number,
        routing_number,
      });

      if (!account) throw new Error("Can't find recipient");

      res.status(200).json({
        data: account,
      });
      return;
    }

    const account = await DB.USER.findOne({
      "account.number": account_number,
    }).select("name _id");

    if (!account) throw new Error("Can't find recipient");

    res.status(200).json({
      data: account,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request.",
    });
  }
});

const verifyOTP = async (req, res, next) => {
  try {
    if (req.user.tfa.status == "inactive")
      throw new Error("Enable two factor to continue");

    if (req.user.otp.direction == "authenticator") {
      const isValid = speakeasy.totp.verify({
        secret: req.user.tfa.token,
        encoding: "base32",
        token: req.body.otp,
      });

      if (!isValid) {
        res.status(500).json({
          e: true,
          msg: "Invalid 2FA token",
        });
        return;
      }

      next();
    } else {
      if (req.user.otp.code.toLowerCase() !== req.body.otp?.toLowerCase()) {
        res.status(500).json({
          e: true,
          msg: "Invalid 2FA token",
        });
        return;
      }

      next();
    }
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
};

// CREATE NEW WIRE TRANSFER
// Router.post("/transfer/create", verifySession, verifyOTP, async (req, res) => {
//   try {
//     let { error, value } = ACCEPTED_TRANSFER_SCHEMA.validate(req.body);
//     if (error) throw new Error(error.details[0].message);

//     const findUserForTransaction = await DB.USER.findById(req.user._id).select(
//       "_id transaction_status failed_transaction_msg"
//     );

//     if (findUserForTransaction?.transaction_status == "fail") {
//       res.status(200).json({
//         msg: findUserForTransaction?.failed_transaction_msg,
//         status: "fail",
//       });
//       return;
//     }

//     // check if the user email or phone already exists
//     let {
//       otp,
//       bankID,
//       receiverID,
//       amount,
//       memo,
//       transfer_type,
//       charges,
//       transfer_from,
//       type,
//     } = value;

//     if (transfer_type == "local") {
//       const findLocalReceiver = await DB.USER.findById(receiverID);

//       if (!findLocalReceiver) throw new Error("Can't find recipient");
//     }

//     if (transfer_type == "wire") {
//       const findWireReceiver = await DB.BANKS.findById(bankID);
//       if (!findWireReceiver) throw new Error("Can't find recipient");
//     }

//     const two_percentage = parseInt(amount) * 0.02;

//     // now minus the amount plus the charge from the user db depending on the transfer_from
//     const char = transfer_type == "wire" ? two_percentage : 0;
//     const cal = parseInt(amount) + char;

//     const useramount = req.user.account.balance[transfer_from];
//     if (cal > useramount) throw new Error("Insufficient balance");
//     // let accountToUpdate = `account.balance${[transfer_from]}`;
//     await DB.USER.findByIdAndUpdate(
//       req.user._id,
//       {
//         $inc: {
//           ["account.balance." + transfer_from]: -cal,
//         },
//       },
//       { new: true }
//     );

//     const ref = generateRef(17);
//     await DB.TRANSACTION.create({
//       uid: req.user._id,
//       ref,
//       bankID,
//       receiverID,
//       amount,
//       memo,
//       transfer_type,
//       transfer_from,
//       charges: char,
//       type,
//       status:
//         findUserForTransaction?.transaction_status == "pending"
//           ? "pending"
//           : "successful",
//     });

//     await DB.NOTIFICATION.create({
//       uid: req.user._id,
//       type: "success",
//       body: {
//         title: "Debit Transaction Successful",
//         // READING AND REPLACING WHERE {{NAME FOR THE REAL NAME}} ALL THE FILE SYSTEM
//         text: `An amount of ${amount} has been debited from your ${transfer_from?.toUpperCase()} account. You can view more details in your transaction history`,
//       },
//     });

//     if (type == "debit" && transfer_type == "local") {
//       await DB.TRANSACTION.create({
//         uid: receiverID,
//         ref,
//         bankID,
//         receiverID: req.user._id,
//         amount,
//         memo,
//         transfer_type,
//         transfer_from,
//         charges: char,
//         type: "credit",
//       });

//       await DB.USER.findByIdAndUpdate(
//         receiverID,
//         {
//           $inc: {
//             ["account.balance." + transfer_from]: parseInt(amount),
//           },
//         },
//         { new: true }
//       );

//       await DB.NOTIFICATION.create({
//         uid: req.user._id,
//         type: "success",
//         body: {
//           title: "Credit Transaction Successful",
//           // READING AND REPLACING WHERE {{NAME FOR THE REAL NAME}} ALL THE FILE SYSTEM
//           text: `An amount of ${amount} has been credited from your ${transfer_from?.toUpperCase()} account. You can view more details in your transaction history`,
//         },
//       });
//     }

//     res.status(200).json({
//       data: "Successful.",
//     });
//   } catch (error) {
//     res.status(500).json({
//       msg: error.message || "unable to complete request.",
//     });
//   }
// });


Router.post('/transfer/init', verifySession, async (req, res) => {
  try {
    // 1. Validate request
    const { error, value } = transferSchema.validate(req.body);
    if (error) {
      return res.status(400).json({
        success: false,
        msg: error.details[0].message
      });
    }



    const {
      beneficiaryId,  // Existing beneficiary
      newBeneficiary, // New beneficiary details
      amount,
      memo,
      saveBeneficiary,
      currency
    } = value;

    const sender = await DB.USER.findById(req.user._id);
    if (!sender) {
      return res.status(404).json({ success: false, msg: 'Sender not found' });
    }


    // 3. Check balance
    if (sender.account.balance[currency] < amount) {
      return res.status(400).json({
        success: false,
        msg: 'Insufficient funds'
      });
    }

    // 2. Handle beneficiary
    let recipient, beneficiary;
    if (beneficiaryId) {
      // Use existing beneficiary
      beneficiary = await DB.BENEFICIARY.findOne({
        _id: beneficiaryId,
        uid: req.user._id
      });
      if (!beneficiary) {
        return res.status(404).json({ success: false, msg: 'Beneficiary not found' });
      }
      // recipient = await DB.BENEFICIARY.findById(beneficiary.receiverID);
    } else if (newBeneficiary) {
      // Validate new beneficiary
      if (!newBeneficiary.accountNumber || !newBeneficiary.name) {
        return res.status(400).json({
          success: false,
          msg: 'New beneficiary requires name and account number'
        });
      }

      // Create temporary recipient (actual user lookup would happen here)
      // recipient = {
      //   _id: new mongoose.Types.ObjectId(), // Temporary ID
      //   name: newBeneficiary.name,
      //   accountNumber: newBeneficiary.accountNumber
      // };

      // Save beneficiary if requested
      if (saveBeneficiary) {
        beneficiary = await DB.BENEFICIARY.create({
          uid: req.user._id,
          name: newBeneficiary.name,
          accountNumber: newBeneficiary.accountNumber,
          bankName: newBeneficiary.bankName,
          type: newBeneficiary.type || 'local',
          ...(newBeneficiary.type === 'local' ? {
            localDetails: {
              routingNumber: newBeneficiary.routingNumber
            }
          } : {
            wireDetails: {
              swiftCode: newBeneficiary.swiftCode,
              country: newBeneficiary.country
            }
          })
        });
      }
    } else {
      return res.status(400).json({
        success: false,
        msg: 'Either beneficiaryId or newBeneficiary must be provided'
      });
    }



    // 4. Determine transaction status
    let status;
    switch (sender.transaction_status) {
      case 'pending': status = 'pending'; break;
      case 'fail': status = 'rejected'; break;
      default: status = 'successful';
    }

    // 5. Create transaction
    const transaction = await DB.TRANSACTION.create({
      uid: req.user._id,
      receiverID: beneficiaryId || new mongoose.Types.ObjectId(),
      amount,
      memo,
      status,
      type: 'debit',
      transfer_type: beneficiary?.type || newBeneficiary?.type || 'local',
      transfer_from: (currency.toLowerCase().replace('o','')), // Default currency
      ...(beneficiary?.bankID && { bankID: beneficiary.bankID }),
      metadata: {
        beneficiaryName: beneficiary?.name || newBeneficiary.name,
        ...(beneficiaryId && { beneficiaryId }),
        ...(!beneficiaryId && { newBeneficiary: true })
      }
    });

    if(sender.transaction_status === 'fail') throw new Error(sender.failed_transaction_msg)





    // For currency-specific balance updates
    const updateUserBalance = async (userId, currency, amount) => {
      const updateQuery = {
        $inc: { [`account.balance.${currency}`]: amount }
      };

      const updatedUser = await DB.USER.findByIdAndUpdate(
        userId,
        updateQuery,
        { new: true }
      );

      if (!updatedUser) {
        throw new Error('User not found');
      }

      return updatedUser;
    };

    // Usage in your transfer route
    if (status === 'successful') {
      // Debit sender
      await updateUserBalance(req.user._id, currency, -amount);
      // send mail here 
      await sendTransactionAlertEmail(sender.name.first,sender.email, 'debit',amount,(currency.toLowerCase().replace('o','')),'****',new Date, memo )
    }

    // 7. Return response
    res.status(201).json({
      success: true,
      msg: `Transfer ${status}`,
      data: {
        transactionId: transaction._id,
        reference: transaction.ref,
        status: transaction.status,
        ...(beneficiary && { beneficiaryId: beneficiary._id }),
        newBalance: sender.account.balance[currency] - amount
      }
    });

  } catch (error) {
    console.error('Transfer error:', error);
    res.status(500).json({
      success: false,
      msg: error.message || 'Transfer failed'
    });
  }
});




// verify tin 
Router.post('/vat/verify', verifySession, async (req,res)=>{
  try {
    let {vatCode} = req.body;
    if(!vatCode) throw new Error('invalid code provided');
    let vatIsCorrect = vatCode.toLowerCase() === req.user?.vat?.code?.toString().toLowerCase()
    if(!vatIsCorrect) throw new Error('Invalid TIN ID provided');
    // if the code is correct then update the status
    await DB.USER.updateOne({_id: req.user._id}, {
        'vat.status': 'paid'
    })
    res.status(201).json({
      data: 'Verification completed'
    })

  } catch (error) {
    res.status(500).json({
      success: false,
      msg: error.message || 'verification failed'
    });
  }
})

module.exports = Router;
