const Joi = require('joi');
const { BENEFICIARY_TYPES, COUNTRY_CODES } = require('../constants'); // You'll need to define these constants

const beneficiaryValidator = Joi.object({
  // Basic info
  name: Joi.string().required().trim().min(2).max(100)
    .messages({
      'string.empty': 'Beneficiary name is required',
      'string.min': 'Beneficiary name must be at least {#limit} characters',
      'string.max': 'Beneficiary name cannot exceed {#limit} characters'
    }),
    
  type: Joi.string().valid(...BENEFICIARY_TYPES).required()
    .messages({
      'any.only': `Type must be one of: ${BENEFICIARY_TYPES.join(', ')}`,
      'any.required': 'Transfer type is required'
    }),
  
  accountNumber: Joi.string().required().trim().min(4).max(34) // IBAN max is 34 chars
    .pattern(/^[A-Z0-9]+$/, 'alphanumeric')
    .messages({
      'string.empty': 'Account number is required',
      'string.pattern.base': 'Account number must be alphanumeric without spaces'
    }),
  
  bankName: Joi.string().required().trim().min(2).max(100)
    .messages({
      'string.empty': 'Bank name is required',
      'string.min': 'Bank name must be at least {#limit} characters',
      'string.max': 'Bank name cannot exceed {#limit} characters'
    }),
  
  // Local transfer fields
  localDetails: Joi.when('type', {
    is: 'local',
    then: Joi.object({
      routingNumber: Joi.string().trim()
        .messages({
          'string.length': 'Routing number must be 9 digits',
          'string.pattern.base': 'Routing number must contain only digits'
        }),
      sortCode: Joi.string().trim().min(6)
        .messages({
          'string.length': 'Sort code must be 6 digits or above',
          'string.pattern.base': 'Sort code must contain only digits'
        }),
      iban: Joi.string().trim().min(13).max(34)
        .messages({
          'string.pattern.base': 'Invalid IBAN format',
          'string.min': 'IBAN must be at least {#limit} characters',
          'string.max': 'IBAN cannot exceed {#limit} characters'
        }),
      // bic: Joi.string().trim().min(8).pattern(/^[A-Z]{6}[A-Z0-9]{2}([A-Z0-9]{3})?$/)
      bic: Joi.string().trim().min(8)
        .messages({
          'string.pattern.base': 'Invalid BIC/SWIFT format',
          'string.length': 'BIC must be must be not less than 8 characters'
        })
    }).or('routingNumber', 'sortCode', 'iban')
      .messages({
        'object.missing': 'Either routing number, sort code, or IBAN is required for local transfers'
      })
  }),
  
  // International transfer fields
  wireDetails: Joi.when('type', {
    is: 'wire',
    then: Joi.object({
      swiftCode: Joi.string()
  .required()
  .trim()
  // .pattern(/^[A-Z]{6}[A-Z0-9]{2}([A-Z0-9]{3})?$/)
  .custom((value, helpers) => {
    if (value.length < 8) {
      return helpers.message('SWIFT code must be not less than 8 characters');
    }
    return value;
  })
  .messages({
    'string.empty': 'SWIFT code is required',
    'string.pattern.base': 'Invalid SWIFT/BIC format. Must start with 6 letters, followed by 2 alphanumeric characters (and optional 3 more for branch)',
    'any.custom': 'SWIFT code must must be not less than 8 characters'
  }),
      bankAddress: Joi.string().required().trim().min(10).max(200)
        .messages({
          'string.empty': 'Bank address is required',
          'string.min': 'Bank address must be at least {#limit} characters',
          'string.max': 'Bank address cannot exceed {#limit} characters'
        }),
      beneficiaryAddress: Joi.string().required().trim().min(10).max(200)
        .messages({
          'string.empty': 'Beneficiary address is required',
          'string.min': 'Beneficiary address must be at least {#limit} characters',
          'string.max': 'Beneficiary address cannot exceed {#limit} characters'
        }),
      country: Joi.string().required().valid(...COUNTRY_CODES).uppercase()
        .messages({
          'any.only': 'Invalid country code',
          'any.required': 'Country is required'
        }),
      intermediaryBank: Joi.object({
        name: Joi.string().trim().min(2).max(100),
        account: Joi.string().trim().min(4).max(34),
        swiftCode: Joi.string().trim().min(8).pattern(/^[A-Z]{6}[A-Z0-9]{2}([A-Z0-9]{3})?$/)
      }).optional()
    }).required()
  }),
  
  // Metadata
  isFavorite: Joi.boolean().default(false),
  lastUsed: Joi.date().iso(),
  
  // Contact info
  email: Joi.string().trim().lowercase().email()
    .messages({
      'string.email': 'Invalid email address'
    }),
  phone: Joi.string().trim().pattern(/^[+]*[(]{0,1}[0-9]{1,4}[)]{0,1}[-\s\./0-9]*$/)
    .messages({
      'string.pattern.base': 'Invalid phone number format'
    }),
  
  // Additional info
  notes: Joi.string().trim().max(500)
    .messages({
      'string.max': 'Notes cannot exceed {#limit} characters'
    }),
  tags: Joi.array().items(Joi.string().trim().max(20))
    .messages({
      'array.base': 'Tags must be an array',
      'string.max': 'Each tag cannot exceed {#limit} characters'
    })
})
  // Validate that either localDetails or wireDetails exists based on type
  .xor('localDetails', 'wireDetails')
  .messages({
    'object.xor': 'Must provide either localDetails or wireDetails based on transfer type'
  })
  // Custom validation for account number based on type
  .custom((value, helpers) => {
    if (value.type === 'local' && value.localDetails?.iban) {
      // Validate IBAN checksum if provided
      if (!isValidIBAN(value.localDetails.iban)) {
        return helpers.message('Invalid IBAN checksum');
      }
    }
    return value;
  });

// Helper function for IBAN validation
function isValidIBAN(iban) {
  // Implement proper IBAN validation logic
  // This is a simplified version - use a proper library in production
  return iban.length >= 15 && iban.match(/^[A-Z]{2}[0-9]{2}[A-Z0-9]{1,30}$/);
}

module.exports = {
  createBeneficiaryValidator: beneficiaryValidator,
  updateBeneficiaryValidator: beneficiaryValidator.fork(
    ['name', 'type', 'accountNumber', 'bankName'], 
    field => field.optional()
  )
};
